"""<p>&nbsp;&nbsp;&nbsp;&nbsp;You will find here all the public classes and methods that you can use to script Hopper with the Python language."""
"""<br/>"""
"""Before using the scripting capabilities of Hopper, you have to understand how Hopper considers its documents."""
"""The main idea is that a document is a set of segments, each of them containing typed bytes."""
"""Indeed, let's consider a standard MachO file. It will contains at least one segment of code (usually named TEXT)"""
"""containing many bytes. Hopper attach some information to each bytes of each segments. At the loading time, all"""
"""bytes are set to the type <b>TYPE_UNDEFINED</b>. Then, starting from the entry point, Hopper will starts to set the type for"""
"""some of them to <b>TYPE_CODE</b>, following the program control flow."""
"""When an instruction needs more than one byte, Hopper will set the type of the first byte to <b>TYPE_CODE</b>, and the following bytes"""
"""to the <b>TYPE_NEXT</b> type."""
"""<br/>"""
"""Using Python, you'll be able to manipulate the segments of disassembled files, retrieving information on bytes types,"""
"""write or read data, change or create name of labels, etc..."""
"""You'll usually starts at retrieving the current document, using the static method <b>Document.getCurrentDocument()</b>.<br/>"""
"""<br/>"""
"""I tried to name the functions in a way that make it easy to find. Fell free to contact me if something seems difficult to achieve,"""
"""or if you have any suggestion."""
"""My e-mail address is <a href="mailto:bsr43@hopperapp.com?subject=Hopper%20Script:%20help%20needed"><i>bsr43@hopperapp.com</i></a>.</p>"""
"""<p>Please note that you can use either, the integrated (very) light Python editor, or use a more convenient editor (like the"""
"""insanely great <i>TextMate</i> editor), but avoid opening the script in both editors at the same time! To use your own editor, simply click on the small gear, underneath the script list. This will"""
"""open a new Finder. When you add, remove or edit files of this directory, Hopper automatically detects it."""
"""</p>"""
import HopperLowLevel

class Procedure:
	"""This class represents a procedure, which is a collection of BasicBlocks."""
	"""Please note that modifying the document (creating new procedure, or deleting an existing one), will result in a possible"""
	"""inconsistancy of the Python's Procedure object representation."""
	def __init__(self,segment_internal,procedure_index):
		self.__segment_internal__ = segment_internal
		self.__procedure_index__ = procedure_index
	def getEntryPoint(self):
		"""Returns the address of the entry point."""
		return HopperLowLevel.getProcedureEntryPoint(self.__segment_internal__,self.__procedure_index__)
	def getBasicBlockCount(self):
		"""Returns the total number of basic blocks."""
		return HopperLowLevel.getBasicBlockCount(self.__segment_internal__,self.__procedure_index__)
	def getBasicBlock(self,index):
		"""Get a BasicBlock object by index."""
		if index < 0 or index >= self.getBasicBlockCount():
			return None
		else:
			return BasicBlock(self,index)
	def getHeapSize(self):
		"""Returns the heap size of the procedure in bytes."""
		return HopperLowLevel.getProcedureHeapSize(self.__segment_internal__,self.__procedure_index__)

class BasicBlock:
	"""A BasicBlock is a set of instructions that is guaranteed to be executed in a whole, if the control flow reach the first instruction."""
	def __init__(self,procedure,basic_block_index):
		self.__procedure__ = procedure
		self.__basic_block_index__ = basic_block_index

	def getProcedure(self):
		"""Returns the Procedure object this BasicBlock belongs to."""
		return self.__procedure__
	def getStartingAddress(self):
		"""Returns the address of the first instruction of the BasicBlock."""
		return HopperLowLevel.getBasicBlockStartingAddress(self.__procedure__.__segment_internal__,self.__procedure__.__procedure_index__,self.__basic_block_index__)
	def getEndingAddress(self):
		"""Returns the address following the last instruction of the BasicBlock."""
		return HopperLowLevel.getBasicBlockEndingAddress(self.__procedure__.__segment_internal__,self.__procedure__.__procedure_index__,self.__basic_block_index__)
	def getSuccessorCount(self):
		"""Returns the number of successors for this BasicBlock."""
		return HopperLowLevel.getBasicBlockSuccessorCount(self.__procedure__.__segment_internal__,self.__procedure__.__procedure_index__,self.__basic_block_index__)
	def getSuccessorIndexAtIndex(self,index):
		"""Returns the BasicBlock index of the Nth successors."""
		return HopperLowLevel.getBasicBlockSuccessorIndex(self.__procedure__.__segment_internal__,self.__procedure__.__procedure_index__,self.__basic_block_index__,index)
	def getSuccessorAddressAtIndex(self,index):
		"""Returns the BasicBlock address of the Nth successors."""
		return HopperLowLevel.getBasicBlockSuccessorAddress(self.__procedure__.__segment_internal__,self.__procedure__.__procedure_index__,self.__basic_block_index__,index)

class Instruction:
	"""This class reprents a disassembled instruction."""
	"""The class defines some constants, like <b>ARCHITECTURE_UNKNOWN</b>, <b>ARCHITECTURE_i386</b> and <b>ARCHITECTURE_X86_64</b>"""

	ARCHITECTURE_UNKNOWN = 0
	ARCHITECTURE_i386 = 1
	ARCHITECTURE_X86_64 = 2
	ARCHITECTURE_ARM = 3
	ARCHITECTURE_ARM_THUMB = 4

	def __init__(self,archi,instr,rawArgs,formattedArgs,cjmp,ijmp,instrLen):
		self.__archi__ = archi
		self.__instr__ = instr
		self.__rawArgs__ = rawArgs;
		self.__formattedArgs__ = formattedArgs
		self.__cjmp__ = cjmp
		self.__ijmp__ = ijmp
		self.__instrLen__ = instrLen

	@staticmethod
	def stringForArchitecture(t):
		"""Helper method that converts one of the architecture value (<b>ARCHITECTURE_UNKNOWN</b>, <b>ARCHITECTURE_i386</b>, <b>ARCHITECTURE_X86_64</b>, <b>ARCHITECTURE_ARM</b> or <b>ARCHITECTURE_ARM_THUMB</b>) to a string value."""
		return {
		Instruction.ARCHITECTURE_UNKNOWN : "unknown",
		Instruction.ARCHITECTURE_i386 : "i386",
		Instruction.ARCHITECTURE_X86_64 : "x86_64",
		Instruction.ARCHITECTURE_ARM : "ARM",
		Instruction.ARCHITECTURE_ARM_THUMB : "ARM (Thumb)",
		}.get(t, "<unknown>")

	def getArchitecture(self):
		"""Returns the architecture."""
		return self.__archi__

	def getInstructionString(self):
		"""Return a strings representing the instruction."""
		return self.__instr__

	def getArgumentCount(self):
		"""Returns the number of argument."""
		return len(self.__rawArgs__)

	def getRawArgument(self,index):
		"""Returns the instruction argument, identified by an index. The argument is not modified by Hopper, and represents the raw ASM argument."""
		if index < 0 or index >= len(self.__rawArgs__):
			return None
		return self.__rawArgs__[index]

	def getFormattedArgument(self,index):
		"""Returns the instruction argument, identified by an index. The argument may have been modified according to the user, or by Hopper if"""
		""" a specific pattern has been detected."""
		if index < 0 or index >= len(self.__formattedArgs__):
			return None
		return self.__formattedArgs__[index]

	def isAnInconditionalJump(self):
		"""Returns True if the instruction represents an inconditional jump."""
		return self.__ijmp__

	def isAConditionalJump(self):
		"""Returns True if the instruction represents a conditional jump."""
		return self.__cjmp__

	def getInstructionLength(self):
		"""Returns the instruction length in byte."""
		return self.__instrLen__

class Segment:
	"""This class represents a segment of a disassembled file."""
	"""The class defines some values that are used as the type"""
	"""of bytes of the disassembled file.<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_UNDEFINED</b> : an undefined byte<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_NEXT</b> : a byte that is part of a larger data type (ex, the second byte of a 4 bytes integer...)<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_BYTE</b> : an integer of a single byte<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_SHORT</b> : an integer of 2 bytes<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_INT</b> : an integer of 4 bytes<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_LONG</b> : an integer of 8 bytes<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_ASCII</b> : an ASCII string<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_UNICODE</b> : an UNICODE string<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_CODE</b> : an instruction<br/>"""
	"""&nbsp;&nbsp;&nbsp;&nbsp;<b>TYPE_PROCEDURE</b> : a procedure<br/>"""
	"""<br/>"""
	"""The class defines the constant <b>BAD_ADDRESS</b> that is returned by some methods when the"""
	"""requested information is incorrect."""

	BAD_ADDRESS=-1

	TYPE_UNDEFINED=0
	TYPE_NEXT=2
	TYPE_BYTE=3
	TYPE_SHORT=4
	TYPE_INT=5
	TYPE_LONG=6
	TYPE_ASCII=7
	TYPE_UNICODE=8
	TYPE_CODE=10
	TYPE_PROCEDURE=11

	@staticmethod
	def stringForType(t):
		"""Helper method that converts one of the type value (<b>TYPE_UNDEFINED</b>, <b>TYPE_NEXT</b>, ...) to a string value."""
		return {
		Segment.TYPE_UNDEFINED : "undefined",
		Segment.TYPE_NEXT : "next",
		Segment.TYPE_BYTE : "byte",
		Segment.TYPE_SHORT : "short",
		Segment.TYPE_INT : "int",
		Segment.TYPE_LONG : "long",
		Segment.TYPE_ASCII : "ascii",
		Segment.TYPE_UNICODE : "unicode",
		Segment.TYPE_CODE : "code",
		Segment.TYPE_PROCEDURE : "procedure",
		}.get(t, "<unknown>")

	def __init__(self,addr):
		self.__internal_segment_addr__ = addr
	def getName(self):
		"""Returns the name of the segment."""
		return HopperLowLevel.getSegmentName(self.__internal_segment_addr__)
	def getStartingAddress(self):
		"""Returns the starting address of the segment."""
		return HopperLowLevel.getSegmentStartingAddress(self.__internal_segment_addr__)
	def getLength(self):
		"""Returns the length, in bytes, of the segment."""
		return HopperLowLevel.getSegmentLength(self.__internal_segment_addr__)
	def getFileOffset(self):
		"""Returns the file offset of the beginning of the segment."""
		return HopperLowLevel.getFileOffset(self.__internal_segment_addr__)
	def getFileOffsetForAddress(self,addr):
		"""Returns the file offset of a particular address."""
		return self.getFileOffset() + addr - self.getStartingAddress()
	def readByte(self,addr):
		"""Read a byte at a given address. Returns False if the byte is read outside of the segment."""
		return HopperLowLevel.readByte(self.__internal_segment_addr__,addr)
	def writeByte(self,addr,value):
		"""Write a byte at a given address. Return True if the writting has succeed."""
		return HopperLowLevel.writeByte(self.__internal_segment_addr__,addr,value)
	def markAsUndefined(self,addr):
		"""Mark the address as being undefined."""
		return HopperLowLevel.markAsUndefined(self.__internal_segment_addr__,addr)
	def markRangeAsUndefined(self,addr,length):
		"""Mark the address range as being undefined."""
		return HopperLowLevel.markRangeAsUndefined(self.__internal_segment_addr__,addr,length)
	def markAsCode(self,addr):
		"""Mark the address as being code."""
		return HopperLowLevel.markAsCode(self.__internal_segment_addr__,addr)
	def markAsProcedure(self,addr):
		"""Mark the address as being a procedure."""
		return HopperLowLevel.markAsProcedure(self.__internal_segment_addr__,addr)
	def markAsDataByteArray(self,addr,count):
		"""Mark the address as being byte array."""
		return HopperLowLevel.markAsDataByteArray(self.__internal_segment_addr__,addr,count)
	def markAsDataShortArray(self,addr,count):
		"""Mark the address as being a short array."""
		return HopperLowLevel.markAsDataShortArray(self.__internal_segment_addr__,addr,count)
	def markAsDataIntArray(self,addr,count):
		"""Mark the address as being an int array."""
		return HopperLowLevel.markAsDataIntArray(self.__internal_segment_addr__,addr,count)
	def isThumbAtAddress(self,addr):
		"""Returns True is instruction at address addr is ARM Thumb mode."""
		return HopperLowLevel.isThumbAtAddress(self.__internal_segment_addr__,addr)
	def setThumbModeAtAddress(self,addr):
		"""Set the Thumb mode at the given address."""
		return HopperLowLevel.setThumbModeAtAddress(self.__internal_segment_addr__,addr)
	def setARMModeAtAddress(self,addr):
		"""Set the ARM mode at the given address."""
		return HopperLowLevel.setARMModeAtAddress(self.__internal_segment_addr__,addr)
	def getTypeAtAddress(self,addr):
		"""Returns the type of the byte at a given address."""
		"""The type can be <b>TYPE_UNDEFINED</b>, <b>TYPE_NEXT</b>, <b>TYPE_BYTE</b>, <b>TYPE_SHORT</b>, <b>TYPE_INT</b>, <b>TYPE_LONG</b>, <b>TYPE_ASCII</b>, <b>TYPE_CODE</b> or <b>TYPE_PROCEDURE</b>."""
		"""The method will returns None if the address is outside the segment."""
		return HopperLowLevel.getTypeAtAddress(self.__internal_segment_addr__,addr)
	def setTypeAtAddress(self,addr,length,typeValue):
		"""Set the type of a byte range."""
		"""The type must be <b>TYPE_UNDEFINED</b>, <b>TYPE_BYTE</b>, <b>TYPE_SHORT</b>, <b>TYPE_INT</b>, <b>TYPE_LONG</b> or <b>TYPE_ASCII</b>."""
		return HopperLowLevel.setTypeAtAddress(self.__internal_segment_addr__,addr,length,typeValue)
	def getNextAddressWithType(self,addr,typeValue):
		"""Returns the next address of a given type."""
		"""The search begins at the given address, so the returned value can be the given address itself."""
		"""If no address are found, the returned value is <b>BAD_ADDRESS</b>."""
		return HopperLowLevel.getNextAddressWithType(self.__internal_segment_addr__,addr,typeValue)
	def disassembleWholeSegment(self):
		"""Disassemble the whole segment."""
		return HopperLowLevel.disassembleWholeSegment(self.__internal_segment_addr__)
	def setNameAtAddress(self,addr,name):
		"""Set the label name at a given address."""
		return (HopperLowLevel.setNameAtAddress(self.__internal_segment_addr__,addr,name) == 1)
	def getNameAtAddress(self,addr):
		"""Get the label name at a given address."""
		return HopperLowLevel.getNameAtAddress(self.__internal_segment_addr__,addr)
	def getCommentAtAddress(self,addr):
		"""Get the prefix comment at a given address."""
		return HopperLowLevel.getCommentAtAddress(self.__internal_segment_addr__,addr)
	def setCommentAtAddress(self,addr,comment):
		"""Set the prefix comment at a given address."""
		return HopperLowLevel.setCommentAtAddress(self.__internal_segment_addr__,addr,comment)
	def getInlineCommentAtAddress(self,addr):
		"""Get the inline comment at a given address."""
		return HopperLowLevel.getInlineCommentAtAddress(self.__internal_segment_addr__,addr)
	def setInlineCommentAtAddress(self,addr,comment):
		"""Set the inline comment at a given address."""
		return HopperLowLevel.setInlineCommentAtAddress(self.__internal_segment_addr__,addr,comment)
	def getInstructionAtAddress(self,addr):
		"""Get the disassembled instruction at a given address."""
		infos = HopperLowLevel.getInstructionAtAddress(self.__internal_segment_addr__,addr)
		if infos == None:
			return None
		return Instruction(infos[0], infos[1], infos[2], infos[3], infos[4], infos[5], infos[6])
	def getReferencesOfAddress(self,addr):
		"""Get the list of address that reference a given address."""
		return HopperLowLevel.getReferencesOfAddress(self.__internal_segment_addr__,addr)
	def getLabelCount(self):
		"""Get the number of named addresses."""
		return HopperLowLevel.getLabelCount(self.__internal_segment_addr__)
	def getLabelName(self,index):
		"""Get a label name by index."""
		return HopperLowLevel.getLabelName(self.__internal_segment_addr__,index)
	def labelIterator(self):
		"""Iterate over all the labels of a segment."""
		index=0
		count=self.getLabelCount()
		while index<count:
			yield self.getLabelName(index)
			index = index + 1
	def getLabelsList(self):
		"""Returns a list with all the label of a segment."""
		return [name for name in labelIterator()]
	def getProcedureCount(self):
		"""Returns the number of procedures that has been defined in this segment."""
		return HopperLowLevel.getProcedureCount(self.__internal_segment_addr__)
	def getProcedureAtIndex(self,index):
		"""Returns the Nth Procedure object of the segment."""
		return Procedure(self.__internal_segment_addr__,index)
	def getProcedureIndexAtAddress(self,address):
		"""Returns the index of the procedure at a given address of the segment, or -1 if there is no procedure defined there."""
		return HopperLowLevel.getProcedureIndexAtAddress(self.__internal_segment_addr__,address)
	def getProcedureAtAddress(self,address):
		"""Returns the Procedure object at a given address of the segment, or None if there is no procedure defined at there."""
		index=self.getProcedureIndexAtAddress(address)
		if index == -1:
			return None
		else:
			return self.getProcedureAtIndex(index)

class Document:
	"""This class represents the disassembled document. A document is a set of segments."""
	def __init__(self,addr):
		self.__internal_document_addr__ = addr
		self.__segments__ = {}
	@staticmethod
	def getCurrentDocument():
		"""Returns the current document."""
		return Document(HopperLowLevel.currentDocument())
	@staticmethod
	def ask(msg):
		"""Open a window containing a text field, and wait for the user to give a string value. Returns the string, or returns None if the Cancel button is hit."""
		return HopperLowLevel.ask(msg)
	@staticmethod
	def message(msg,buttons):
		"""Open a window containing a text field and a set of buttons. The 'Buttons' parameter is a list of strings. The function returns the index of the clicked button."""
		return HopperLowLevel.message(msg,buttons)
	def assemble(self,instr,address,intel):
		"""Assemble an instruction, and returns the bytes as an array. The instruction is NOT injected in the document: the address given to the function is used to encode the instruction. You wan use the writeByte method to inject an assembled instruction."""
		"""The first argument is the instruction to be assembled. The second is the address of the instruction. The last parameter is a boolean value indicating if the instruction use the Intel syntax or the AT&T syntax."""
		return HopperLowLevel.assemble(self.__internal_document_addr__,instr,address,1 if intel else 0)
	def log(self,msg):
		"""Display a string message into the log window of the document."""
		HopperLowLevel.log(self.__internal_document_addr__,msg)
	def getSegmentCount(self):
		"""Returns the number of segment the document contains."""
		return HopperLowLevel.getSegmentCount(self.__internal_document_addr__);
	def getSegment(self,index):
		"""Returns a segment by index. The returned object is an instance of the Segment class. If the index of not in the range [0;count[, the function returns None."""
		if not self.__segments__.has_key(index):
			addr = HopperLowLevel.getSegmentAddress(self.__internal_document_addr__, index)
			if addr == 0:
				return None
			self.__segments__[index] = Segment(addr)
		return self.__segments__[index]
	def getSegmentsList(self):
		"""Returns a list containing all the segments."""
		return [self.getSegment(x) for x in xrange(self.getSegmentCount())]
	def getCurrentColumn(self):
		"""Returns the column where the cursor currently is."""
		return HopperLowLevel.getCurrentColumn(self.__internal_document_addr__)
	def getSegmentIndexAtAddress(self,addr):
		"""Returns the segment index for a particular address."""
		return HopperLowLevel.getSegmentIndexAtAddress(self.__internal_document_addr__,addr)
	def getSegmentAtAddress(self,addr):
		"""Returns the segment for a particular address."""
		idx=self.getSegmentIndexAtAddress(addr)
		if idx == -1:
			return None
		return self.getSegment(idx)
	def getCurrentSegmentIndex(self):
		"""Returns the segment index where the cursor is. Returns -1 if the current segment cannot be located."""
		return HopperLowLevel.getCurrentSegmentIndex(self.__internal_document_addr__)
	def getCurrentSegment(self):
		"""Returns the segment where the cursor is. Returns None if the current segment cannot be located."""
		index = HopperLowLevel.getCurrentSegmentIndex(self.__internal_document_addr__)
		if index == -1:
			return None
		else:
			return self.getSegment(index)
	def getCurrentAddress(self):
		"""Returns the address where the cursor currently is."""
		return HopperLowLevel.getCurrentAddress(self.__internal_document_addr__)
	def getSelectionAddressRange(self):
		"""Returns a list, containing two addresses. Those address represents the range of bytes covered by the selection."""
		return HopperLowLevel.getSelectionAddressRange(self.__internal_document_addr__)
	def moveCursorAtAddress(self,addr):
		"""Move the cursor at a given address."""
		HopperLowLevel.moveCursorAtAddress(self.__internal_document_addr__,addr)
	def selectAddressRange(self,addrRange):
		"""Select a range of byte. The awaited argument is a list containing exactly two address."""
		HopperLowLevel.selectAddressRange(self.__internal_document_addr__,addrRange[0],addrRange[1])
	def is64Bits(self):
		"""Returns True if the disassembled document is interpreted as a 64 bits binary."""
		return HopperLowLevel.is64Bits(self.__internal_document_addr__);
	def getEntryPoint(self):
		"""Returns the entry point of the document."""
		return HopperLowLevel.getEntryPoint(self.__internal_document_addr__)
	def moveCursorAtEntryPoint(self):
		"""Move the cursor at the entry point."""
		self.moveCursorAtAddress(self.getEntryPoint())
	def getHighlightedWord(self):
		"""Returns the word that is currently highlighted in the assembly view."""
		return HopperLowLevel.getHighlightedWord(self.__internal_document_addr__)
	def setNameAtAddress(self,addr,name):
		"""Set the label name at a given address."""
		seg=self.getSegmentAtAddress(addr)
		if seg != None:
			return seg.setNameAtAddress(addr,name)
		return False
	def getNameAtAddress(self,addr):
		"""Get the label name at a given address."""
		seg=self.getSegmentAtAddress(addr)
		if seg != None:
			return seg.getNameAtAddress(addr)
		return None
	def getAddressForName(self,name):
		"""Get the address associated to a given name."""
		return HopperLowLevel.getAddressForName(self.__internal_document_addr__,name)
	def refreshView(self):
		"""Force the assembly view to be refresh."""
		HopperLowLevel.refreshView(self.__internal_document_addr__)
	def moveCursorOneLineDown(self):
		"""Move the current line down, and remove the multiselection if needed. Returns True if cursor moved."""
		return HopperLowLevel.moveCursorOneLineDown(self.__internal_document_addr__)
	def moveCursorOneLineUp(self):
		"""Move the current line up, and remove the multiselection if needed. Returns True if cursor moved."""
		return HopperLowLevel.moveCursorOneLineUp(self.__internal_document_addr__)
	def getRawSelectedLines(self):
		"""Returns a list of strings corresponding to the current selection."""
		return HopperLowLevel.getRawSelectedLines(self.__internal_document_addr__)
