<!-- 2025.08.19 -->

# August 2025: Bug fixes and stability

This release has mostly been focused on fixing bugs and improving stability over the long term.

## 🐛 Bugs

The migration to Visual Studio 2022 unfortunately caused some serious issues to slip through. The following bugs were fixed:

- Systems with older versions of the [Visual C++ Redistributable](https://learn.microsoft.com/en-us/cpp/windows/latest-supported-vc-redist?view=msvc-170) would crash when you started debugging.
- Pattern finding was completely broken.
- Systems with AVX-512 would crash when using x32dbg.
- CPUs AVX support would always show 0 for XMM registers (thanks to @CXVUSER).

To prevent critical issues like this from slipping through we are working on an automated test system. The _headless_ version of x64dbg introduced in the previous release was a first step in this direction, but much more work is required. This release also introduced support for [AddressSanitizer](https://learn.microsoft.com/en-us/cpp/sanitizers/asan), which helps prevent memory safety issues.

## ❓ Documentation

The [documentation](https://help.x64dbg.com) has now been moved to the `docs` folder in the x64dbg repository. This streamlines development, but also makes the project much more LLM-friendly. Practically-speaking this means you can now ask questions about x64dbg in [DeepWiki](https://deepwiki.com/x64dbg/x64dbg) and expect reasonably-useful results!

[![Ask DeepWiki](data:image/;base64,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)](https://deepwiki.com/x64dbg/x64dbg)

Some example questions:

- [Can I log the call stack while tracing?](https://deepwiki.com/search/is-there-any-way-to-access-the_bfa81d28-091b-4d93-84e7-a1e8a89830d2)
- [Can I get a list of available commands?](https://deepwiki.com/search/give-me-a-list-of-available-co_5316fb4e-b0a9-4984-8ea9-819194c7c247)
- [How to set a conditional breakpoint on an RVA?](https://deepwiki.com/search/relevantcontextthis-query-was_08df641e-ec87-49e9-b73b-facb193f5bc3)
- [How to automate patching steps?](https://deepwiki.com/search/how-to-automate-the-following_18ec3f69-358f-4b9d-bfef-8df43b93e506)

Obviously this is no replacement for reading the documentation, but it might at least point you to some concepts to research!

## 🔎 Tracing

There have been quite a lot of updates to the [trace recording](https://help.x64dbg.com/en/latest/introduction/ConditionalTracing.html) feature over the past few months by @torusrxxx. For those not familiar, trace recording allows you to capture the registers and memory accesses for every step of the program. Think [Time Travel Debugging](https://learn.microsoft.com/en-us/windows-hardware/drivers/debuggercmds/time-travel-debugging-record), but on a much smaller scale. This feature can be helpful trying to understand the data flow or figuring out where a specific value first pops up when executing a function.

To start a trace recording, use _Help -> Trace into..._ and then enable the _Record trace_ checkbox:

![](data:image/;base64,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)

You can see a reconstruction of the memory/stack (best effort, might be slow on larger traces):

![](data:image/;base64,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)

This version introduces support to search for intermodular calls in the trace, which can be useful to understand where APIs get called. A blog post with more details will follow!

## 🤖 Scripting

**❗ You might want to check out [Analysis at Scale with x64dbg Automate](https://x64dbg.com/blog/2025/03/04/analysis-at-scale-with-x64dbg-automate.html), which introduces a Python library to automate x64dbg. There is also an [MCP Server](https://github.com/cycraft-corp/BinaryAnalysisMCPs) built on top of it, which allows LLMs to interact with x64dbg.**

The builtin scripting feature has always been a bit lackluster, but the need for testing automation has pushed things forward. Improved command line handling was introduced by @peace-maker and you can now specify a script to execute on debuggee startup:

```sh
  x64dbg {OPTIONS} [filename] -- [extra arguments]

    x64dbg

  OPTIONS:

    -workingDir  Current working directory of new process.
    Defaults to current working directory if not specified. (optional)
    -pid         Process ID to attach to. (optional)
    -tid         Thread Identifier (TID) of the thread to resume
    after attaching (PLMDebug). (optional)
    -event       Handle to an Event Object to signal (JIT). (optional)
    -c           Command to execute Specifies the initial
    debugger command to run at start-up. (optional)
    -cf          Specifies the path and name of a script file.
    This script file is executed immediately on startup. (optional)
    -p           Alias for -pid. (optional)
    -a           Alias for -pid. (optional)
    -e           Alias for -event (optional)
    -help        Show this message. (optional)
```

The scripting system has been rewritten from scratch to execute all of the script commands on an event loop, which ensures everything executes in a deterministic order. A lot of bugs have been fixed too, which enables more complex scripting scenarios:

```c
fn = RtlEnterCriticalSection
bp fn
SetBreakpointCommand fn, "scriptcmd call analyze"
ret

analyze:
  SpinCount = ReadPtr(rcx + 0x20)
  log "SpinCount: {u:SpinCount}"
  ret
```

The script above sets a breakpoint on `RtlEnterCriticalSection`. It uses a [breakpoint command](https://help.x64dbg.com/en/latest/introduction/ConditionalBreakpoint.html) to execute the `analyze` function in the script every time the breakpoint hits, and performs some logging. For testing we can use the _headless_ command line tool with test executables and scripts to make sure everything works as expected.

## 🐧 Cross Platform

Thanks to @eldarkg you can now [build x64dbg in Wine](https://github.com/x64dbg/x64dbg/blob/development/docs/COMPILE-linux.md) with a custom CMake toolchain. Additionally the first iteration of the `cross-platform` branch has been merged. Slowly the GUI components will be ported to an `x64dbg::widgets` library that can be reused for other projects.

## 🤝 Community
x64dbg is a community-driven project, and this release would not have been possible without the incredible work of our contributors. A huge thank you to everyone who contributed code, reported bugs, and helped shape this release ❤️

We recently revamped the Discord community, which you can join below:

[![](data:image/;base64,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)](https://discord.x64dbg.com)

Other platforms are synchronized with the `#general` channel:

[![Slack](data:image/;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI5MCIgaGVpZ2h0PSIyMCIgcm9sZT0iaW1nIiBhcmlhLWxhYmVsPSJjaGF0OiBvbiBTbGFjayI+PHRpdGxlPmNoYXQ6IG9uIFNsYWNrPC90aXRsZT48bGluZWFyR3JhZGllbnQgaWQ9InMiIHgyPSIwIiB5Mj0iMTAwJSI+PHN0b3Agb2Zmc2V0PSIwIiBzdG9wLWNvbG9yPSIjYmJiIiBzdG9wLW9wYWNpdHk9Ii4xIi8+PHN0b3Agb2Zmc2V0PSIxIiBzdG9wLW9wYWNpdHk9Ii4xIi8+PC9saW5lYXJHcmFkaWVudD48Y2xpcFBhdGggaWQ9InIiPjxyZWN0IHdpZHRoPSI5MCIgaGVpZ2h0PSIyMCIgcng9IjMiIGZpbGw9IiNmZmYiLz48L2NsaXBQYXRoPjxnIGNsaXAtcGF0aD0idXJsKCNyKSI+PHJlY3Qgd2lkdGg9IjMzIiBoZWlnaHQ9IjIwIiBmaWxsPSIjNTU1Ii8+PHJlY3QgeD0iMzMiIHdpZHRoPSI1NyIgaGVpZ2h0PSIyMCIgZmlsbD0iI2UwNWQ0NCIvPjxyZWN0IHdpZHRoPSI5MCIgaGVpZ2h0PSIyMCIgZmlsbD0idXJsKCNzKSIvPjwvZz48ZyBmaWxsPSIjZmZmIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBmb250LWZhbWlseT0iVmVyZGFuYSxHZW5ldmEsRGVqYVZ1IFNhbnMsc2Fucy1zZXJpZiIgdGV4dC1yZW5kZXJpbmc9Imdlb21ldHJpY1ByZWNpc2lvbiIgZm9udC1zaXplPSIxMTAiPjx0ZXh0IGFyaWEtaGlkZGVuPSJ0cnVlIiB4PSIxNzUiIHk9IjE1MCIgZmlsbD0iIzAxMDEwMSIgZmlsbC1vcGFjaXR5PSIuMyIgdHJhbnNmb3JtPSJzY2FsZSguMSkiIHRleHRMZW5ndGg9IjIzMCI+Y2hhdDwvdGV4dD48dGV4dCB4PSIxNzUiIHk9IjE0MCIgdHJhbnNmb3JtPSJzY2FsZSguMSkiIGZpbGw9IiNmZmYiIHRleHRMZW5ndGg9IjIzMCI+Y2hhdDwvdGV4dD48dGV4dCBhcmlhLWhpZGRlbj0idHJ1ZSIgeD0iNjA1IiB5PSIxNTAiIGZpbGw9IiMwMTAxMDEiIGZpbGwtb3BhY2l0eT0iLjMiIHRyYW5zZm9ybT0ic2NhbGUoLjEpIiB0ZXh0TGVuZ3RoPSI0NzAiPm9uIFNsYWNrPC90ZXh0Pjx0ZXh0IHg9IjYwNSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iNDcwIj5vbiBTbGFjazwvdGV4dD48L2c+PC9zdmc+)](https://slack.x64dbg.com) [![Gitter](data:image/;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI5MiIgaGVpZ2h0PSIyMCIgcm9sZT0iaW1nIiBhcmlhLWxhYmVsPSJjaGF0OiBvbiBHaXR0ZXIiPjx0aXRsZT5jaGF0OiBvbiBHaXR0ZXI8L3RpdGxlPjxsaW5lYXJHcmFkaWVudCBpZD0icyIgeDI9IjAiIHkyPSIxMDAlIj48c3RvcCBvZmZzZXQ9IjAiIHN0b3AtY29sb3I9IiNiYmIiIHN0b3Atb3BhY2l0eT0iLjEiLz48c3RvcCBvZmZzZXQ9IjEiIHN0b3Atb3BhY2l0eT0iLjEiLz48L2xpbmVhckdyYWRpZW50PjxjbGlwUGF0aCBpZD0iciI+PHJlY3Qgd2lkdGg9IjkyIiBoZWlnaHQ9IjIwIiByeD0iMyIgZmlsbD0iI2ZmZiIvPjwvY2xpcFBhdGg+PGcgY2xpcC1wYXRoPSJ1cmwoI3IpIj48cmVjdCB3aWR0aD0iMzMiIGhlaWdodD0iMjAiIGZpbGw9IiM1NTUiLz48cmVjdCB4PSIzMyIgd2lkdGg9IjU5IiBoZWlnaHQ9IjIwIiBmaWxsPSJsaWdodHNlYWdyZWVuIi8+PHJlY3Qgd2lkdGg9IjkyIiBoZWlnaHQ9IjIwIiBmaWxsPSJ1cmwoI3MpIi8+PC9nPjxnIGZpbGw9IiNmZmYiIHRleHQtYW5jaG9yPSJtaWRkbGUiIGZvbnQtZmFtaWx5PSJWZXJkYW5hLEdlbmV2YSxEZWphVnUgU2FucyxzYW5zLXNlcmlmIiB0ZXh0LXJlbmRlcmluZz0iZ2VvbWV0cmljUHJlY2lzaW9uIiBmb250LXNpemU9IjExMCI+PHRleHQgYXJpYS1oaWRkZW49InRydWUiIHg9IjE3NSIgeT0iMTUwIiBmaWxsPSIjMDEwMTAxIiBmaWxsLW9wYWNpdHk9Ii4zIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgdGV4dExlbmd0aD0iMjMwIj5jaGF0PC90ZXh0Pjx0ZXh0IHg9IjE3NSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iMjMwIj5jaGF0PC90ZXh0Pjx0ZXh0IGFyaWEtaGlkZGVuPSJ0cnVlIiB4PSI2MTUiIHk9IjE1MCIgZmlsbD0iIzAxMDEwMSIgZmlsbC1vcGFjaXR5PSIuMyIgdHJhbnNmb3JtPSJzY2FsZSguMSkiIHRleHRMZW5ndGg9IjQ5MCI+b24gR2l0dGVyPC90ZXh0Pjx0ZXh0IHg9IjYxNSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iNDkwIj5vbiBHaXR0ZXI8L3RleHQ+PC9nPjwvc3ZnPg==)](https://gitter.im/x64dbg/x64dbg) [![Matrix](data:image/;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI5NCIgaGVpZ2h0PSIyMCIgcm9sZT0iaW1nIiBhcmlhLWxhYmVsPSJjaGF0OiBvbiBNYXRyaXgiPjx0aXRsZT5jaGF0OiBvbiBNYXRyaXg8L3RpdGxlPjxsaW5lYXJHcmFkaWVudCBpZD0icyIgeDI9IjAiIHkyPSIxMDAlIj48c3RvcCBvZmZzZXQ9IjAiIHN0b3AtY29sb3I9IiNiYmIiIHN0b3Atb3BhY2l0eT0iLjEiLz48c3RvcCBvZmZzZXQ9IjEiIHN0b3Atb3BhY2l0eT0iLjEiLz48L2xpbmVhckdyYWRpZW50PjxjbGlwUGF0aCBpZD0iciI+PHJlY3Qgd2lkdGg9Ijk0IiBoZWlnaHQ9IjIwIiByeD0iMyIgZmlsbD0iI2ZmZiIvPjwvY2xpcFBhdGg+PGcgY2xpcC1wYXRoPSJ1cmwoI3IpIj48cmVjdCB3aWR0aD0iMzMiIGhlaWdodD0iMjAiIGZpbGw9IiM1NTUiLz48cmVjdCB4PSIzMyIgd2lkdGg9IjYxIiBoZWlnaHQ9IjIwIiBmaWxsPSIjYTRhNjFkIi8+PHJlY3Qgd2lkdGg9Ijk0IiBoZWlnaHQ9IjIwIiBmaWxsPSJ1cmwoI3MpIi8+PC9nPjxnIGZpbGw9IiNmZmYiIHRleHQtYW5jaG9yPSJtaWRkbGUiIGZvbnQtZmFtaWx5PSJWZXJkYW5hLEdlbmV2YSxEZWphVnUgU2FucyxzYW5zLXNlcmlmIiB0ZXh0LXJlbmRlcmluZz0iZ2VvbWV0cmljUHJlY2lzaW9uIiBmb250LXNpemU9IjExMCI+PHRleHQgYXJpYS1oaWRkZW49InRydWUiIHg9IjE3NSIgeT0iMTUwIiBmaWxsPSIjMDEwMTAxIiBmaWxsLW9wYWNpdHk9Ii4zIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgdGV4dExlbmd0aD0iMjMwIj5jaGF0PC90ZXh0Pjx0ZXh0IHg9IjE3NSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iMjMwIj5jaGF0PC90ZXh0Pjx0ZXh0IGFyaWEtaGlkZGVuPSJ0cnVlIiB4PSI2MjUiIHk9IjE1MCIgZmlsbD0iIzAxMDEwMSIgZmlsbC1vcGFjaXR5PSIuMyIgdHJhbnNmb3JtPSJzY2FsZSguMSkiIHRleHRMZW5ndGg9IjUxMCI+b24gTWF0cml4PC90ZXh0Pjx0ZXh0IHg9IjYyNSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iNTEwIj5vbiBNYXRyaXg8L3RleHQ+PC9nPjwvc3ZnPg==)](https://riot.im/app/#/room/#x64dbg:matrix.org) [![IRC](data:image/;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI4MCIgaGVpZ2h0PSIyMCIgcm9sZT0iaW1nIiBhcmlhLWxhYmVsPSJjaGF0OiBvbiBJUkMiPjx0aXRsZT5jaGF0OiBvbiBJUkM8L3RpdGxlPjxsaW5lYXJHcmFkaWVudCBpZD0icyIgeDI9IjAiIHkyPSIxMDAlIj48c3RvcCBvZmZzZXQ9IjAiIHN0b3AtY29sb3I9IiNiYmIiIHN0b3Atb3BhY2l0eT0iLjEiLz48c3RvcCBvZmZzZXQ9IjEiIHN0b3Atb3BhY2l0eT0iLjEiLz48L2xpbmVhckdyYWRpZW50PjxjbGlwUGF0aCBpZD0iciI+PHJlY3Qgd2lkdGg9IjgwIiBoZWlnaHQ9IjIwIiByeD0iMyIgZmlsbD0iI2ZmZiIvPjwvY2xpcFBhdGg+PGcgY2xpcC1wYXRoPSJ1cmwoI3IpIj48cmVjdCB3aWR0aD0iMzMiIGhlaWdodD0iMjAiIGZpbGw9IiM1NTUiLz48cmVjdCB4PSIzMyIgd2lkdGg9IjQ3IiBoZWlnaHQ9IjIwIiBmaWxsPSJwdXJwbGUiLz48cmVjdCB3aWR0aD0iODAiIGhlaWdodD0iMjAiIGZpbGw9InVybCgjcykiLz48L2c+PGcgZmlsbD0iI2ZmZiIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZm9udC1mYW1pbHk9IlZlcmRhbmEsR2VuZXZhLERlamFWdSBTYW5zLHNhbnMtc2VyaWYiIHRleHQtcmVuZGVyaW5nPSJnZW9tZXRyaWNQcmVjaXNpb24iIGZvbnQtc2l6ZT0iMTEwIj48dGV4dCBhcmlhLWhpZGRlbj0idHJ1ZSIgeD0iMTc1IiB5PSIxNTAiIGZpbGw9IiMwMTAxMDEiIGZpbGwtb3BhY2l0eT0iLjMiIHRyYW5zZm9ybT0ic2NhbGUoLjEpIiB0ZXh0TGVuZ3RoPSIyMzAiPmNoYXQ8L3RleHQ+PHRleHQgeD0iMTc1IiB5PSIxNDAiIHRyYW5zZm9ybT0ic2NhbGUoLjEpIiBmaWxsPSIjZmZmIiB0ZXh0TGVuZ3RoPSIyMzAiPmNoYXQ8L3RleHQ+PHRleHQgYXJpYS1oaWRkZW49InRydWUiIHg9IjU1NSIgeT0iMTUwIiBmaWxsPSIjMDEwMTAxIiBmaWxsLW9wYWNpdHk9Ii4zIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgdGV4dExlbmd0aD0iMzcwIj5vbiBJUkM8L3RleHQ+PHRleHQgeD0iNTU1IiB5PSIxNDAiIHRyYW5zZm9ybT0ic2NhbGUoLjEpIiBmaWxsPSIjZmZmIiB0ZXh0TGVuZ3RoPSIzNzAiPm9uIElSQzwvdGV4dD48L2c+PC9zdmc+)](https://web.libera.chat/#x64dbg)

## ❤️ Sponsors
This project is also made possible by the generous support of sponsors. We would like to give a special shout out to the following sponsors:
- [**FLOSS/fund**](https://floss.fund) - Funding for Free and Open Source projects.
- [**SEKTOR7 Institute**](https://institute.sektor7.net/)
- [**InfinityCurve**](https://infinitycurve.org/)

Also many thanks to my active sponsors: @LabGuy94, @grant-h, @MorpheusH3x, @re-jevi, @adam-the, @verdeckt, @emesare, @daaximus, @stevemk14ebr, @as0ni, @sedrakpc, @Dan0xE, @Invoke-RE, @leandrofroes, @shu-tom, @buzzer-re, @expend20, @crudd, @clayne, @fr0zenbag, @dzzie

**If you find x64dbg valuable in your work, please consider [becoming a sponsor](https://github.com/sponsors/mrexodia?metadata_source=releases). Your support directly funds development and helps us continue to build the future of debugging.** For companies we also offer custom services depending on the level of sponsorship, please [reach out](mailto:contact@x64dbg.com) for more details!

## 📜 Changelog
- InitializeSignatureCheck now loads msvcp140.dll
- Always load VC runtime from application directory #3616
- Move https://github.com/x64dbg/docs into this repo
- Fix snapshot timestamp format
- Fix AVX-512 crash on 32-bit
- Exclude unnecessary OpenGL libraries from snapshot
- Fix bug with string formatting
- Fix ambiguous operand size when hiding pointer sizes by @kalhotky in #3620
- Display style values refactor and MASM style by @kalhotky in #3617
- Revert "Fixed inverted logic" by @mrexodia in #3622
- Improve developer experience when forking
- Improve CONTRIBUTING guide
- Another attempt to fix msvcp140 loading #3624
- Unbreak x32dbg after last commit
- Fixed pattern search by @OldGamesCracking in #3628
- Add back -DX64DBG, closes #3634
- Update resources translations by @torusrxxx in #3562
- Faster disasmgetstringatwrapper by @d2k2-git in #3625
- Fix deleting breakpoint at module base address by @kalhotky in #3629
- Fix symbolic name help URL encoding by @kalhotky in #3630
- Fix fastresume hiding other summary text by @kalhotky in #3631
- Disasm value notation by @kalhotky in #3640
- Fix incorrect "None" label for disabled trace coverage in shortcuts by @adityaxa in #3638
- Refactor QZydis instance in Bridge class by @mrexodia in #3641
- Automatically cancel previous workflow runs on the same branch
- align char buffer sizes with defined limit in GuiGetDisassembly by @d2k2-git in #3627
- Fix autocomment precedence by @kalhotky in #3632
- Fix mnemonic brief column #3510, #3509
- MSVC build under linux wine by @eldarkg in #3657
- Use library for command line parsing by @peace-maker in #3516
- Fix headless initialization and thread model
- msvc-wine: Fix: LINK : fatal error LNK1158: cannot run 'rc.exe' by @eldarkg in #3660
- Do not produce an import library for exe target #3663
- Add clang-tidy configuration
- Show struct widget in DisplayTypeDialog
- Misc improvements
- Fix bug where disabling and reenabling GUI updates didn't update the GUI properly
- Make breakpoint silent if breakpoint command execution fails
- JobQueue.await now recursively calls job function when called on worker thread
- Greatly improve script state handling
- Export table to tab-separated values by @torusrxxx in #3666
- Support adjusting font size via ctrl+wheel in RegistersView by @torusrxxx in #3665
- Intermodular calls: add column for module name (#3664)
- Properly synchronize breakpoint commands and script state
- Allow using VirtualQueryEx in MemIsCodePage and enable it everywhere
- Msvc wine build x64 architecture (#3669)
- Patch file: fix wine usage by @eldarkg in #3679
- Unity build by @mrexodia in #3674
- Follow in stack in trace registers view; fix shortcuts in registers view by @torusrxxx in #3670
- Search for intermodular calls in trace by @torusrxxx in #3672
- Goto dialog: add completion for labels by @dabeibao in #3681
- Upgrade \_WIN32\_WINNT to Windows 7 by @torusrxxx in #3668
- Fix code formatting
- Remove dbghelp/DeviceNameResolver/TitanEngine from pluginsdk #3412
- Add some cross platform example projects by @mrexodia in #3224
- Simplify licenses sections of the docs to use links instead
- Fix struct view refreshing and add error message when connection fails
- Add support for sanitizers and fix a crash
- Update debug engines to 2025.08.18
- Release notes dialog by @mrexodia in #3686

Full commit history: [`2025.06.30...2025.08.19`](https://github.com/x64dbg/x64dbg/compare/2025.06.30...2025.08.19)


<!-- 2025.07.04 -->

# June 2025: Type System and Modernization
We're excited to announce a major new release of x64dbg, the open-source user mode debugger for Windows. For those new to the project, x64dbg is designed to make reverse engineering and malware analysis faster and more intuitive. This release marks a significant step forward, overhauling our core type system and modernizing our entire toolchain to bring you a more powerful and stable debugging experience.

**NOTE**: There were some minor issues that made the 2025.06.30 snapshot unusable (signing issue causing a Windows Defender false positive, AVX-512 support crashed in x32dbg and the VC runtime was not loaded correctly). Because of this that snapshot has been deleted and a fixed one has been put here.

## ✨ Revamped Type System
In previous versions, analyzing data structures was a tedious, manual process and many features were not supported. This release adds support for bitfields, enums and anonymous types, which allows all types in the Windows SDK to be represented and displayed.

The [ManyTypes](https://github.com/snowsnowsnows/ManyTypes) plugin by @snowsnowsnows (who also drove this revamp) allows you to import C header files and see the results directly in x64dbg. While we plan to streamline this workflow even further in future updates, this is a huge leap in making data inspection easier.

This isn't just about convenience; it's about speed. We've introduced **drastic performance improvements** to the struct widget, so you can now browse deeply nested pointers and large data structures without the lag.

https://github.com/user-attachments/assets/973d101c-5dab-4f66-ac10-fc68873d52f0

We’ve also added a host of quality-of-life improvements:
- **Interactive type selection**: While selecting the type you will instantly see what the data looks like, enabling a more interactive workflow.
- **Smarter Displays:** Character arrays are now automatically rendered as strings, saving you an extra step.
- **Better Integration:** You can now invoke the "Display type" action directly from the register and stack views, making it easier than ever to inspect data on the fly.

## 💻 AVX-512 and Half-Float Support
This release introduces support for the latest CPU instruction sets, ensuring you can analyze even the most modern applications.
- **AVX-512 Support:** You now have the power to debug and analyze code that leverages the AVX-512 instruction set, a critical feature for high-performance computing and complex malware.
- **Half-Float Support:** We've added support for 16-bit half-precision floating-point numbers in the dump, a feature especially useful when reversing graphics applications or machine learning models.

## ⚡ Small Changes, Big Impact
Sometimes it’s the little things that count. Based on your feedback, we've added several small but mighty workflow enhancements:
- **Copy Graph Image:** You can now copy the function graph directly to your clipboard, making it effortless to share your findings or add visuals to your reports.
- **Copy Calling Convention Arguments:** A new context menu option lets you quickly copy function arguments, streamlining the process of reconstructing code or documenting function calls.
- **Focus on Window in Handles View:** You can now bring a target window to the foreground directly from the handles view, which is incredibly useful for UI-heavy debugging.
- **Revamped Launcher**: The launcher now has checkboxes and an uninstall option.

![image](data:image/;base64,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) ![image](data:image/;base64,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)

## 🚀 Modern Tooling and a New Release Cycle
We've completed a monumental migration of our entire build system to **Visual Studio 2022 and CMake**. This massive undertaking provides a more stable foundation, makes it easier for new developers to contribute, and accelerates the pace of future improvements. The [compilation instructions](https://github.com/x64dbg/x64dbg/wiki/Compiling-the-whole-project) were cut down from _13 steps to a single one._

There has also been movement on a _headless version of x64dbg_. The main focus for now is automated testing, but eventually it will be expanded to allow [headless automations](https://x64dbg.com/blog/2025/03/04/analysis-at-scale-with-x64dbg-automate.html).

Starting with this release, we will be using **[CalVer](https://calver.org)** (Calendar Versioning) with proper GitHub releases and tags. This will keep our releases more organized and allow package managers and users to reference specific versions more easily. Additionally Windows XP is [no longer supported](https://github.com/x64dbg/x64dbg/wiki/Transition) and for Windows 7/8.1 you will get an unskippable deprecation warning:

![image](data:image/;base64,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)

## 🔮 Looking Ahead: A Cross-Platform Future
We're excited to share a glimpse into our experimental [`cross-platform`](https://github.com/x64dbg/x64dbg/pull/3224) branch. This is our testbed for separating core GUI widgets into a reusable library, with the goal of bringing x64dbg's powerful tools to all major platforms (Windows, macOS, and Linux).

Current experimental tools include:
- A simple **hex viewer** that uses the [ImHex Pattern Language](https://github.com/WerWolv/PatternLanguage) to visualize data structures.
- A **minidump viewer** to browse `.dmp` files on any platform.
- A **remote table** tool that showcases fetching data over a high-latency network.

![image](data:image/;base64,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)

These tools are still in early development and not part of the release, but they represent a critical step toward a more versatile and platform-independent future for our components.

## 🤝 Community
x64dbg is a community-driven project, and this release would not have been possible without the incredible work of our contributors. A huge thank you to everyone who contributed code, reported bugs, and helped shape this release ❤️

We recently revamped the Discord community, which you can join below:

[![](data:image/;base64,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)](https://discord.x64dbg.com)

You can also get the `XDBG` tag after joining the server:

![image](data:image/;base64,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)

Other platforms are synchronized with the `#general` channel:

[![Slack](data:image/;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI5MCIgaGVpZ2h0PSIyMCIgcm9sZT0iaW1nIiBhcmlhLWxhYmVsPSJjaGF0OiBvbiBTbGFjayI+PHRpdGxlPmNoYXQ6IG9uIFNsYWNrPC90aXRsZT48bGluZWFyR3JhZGllbnQgaWQ9InMiIHgyPSIwIiB5Mj0iMTAwJSI+PHN0b3Agb2Zmc2V0PSIwIiBzdG9wLWNvbG9yPSIjYmJiIiBzdG9wLW9wYWNpdHk9Ii4xIi8+PHN0b3Agb2Zmc2V0PSIxIiBzdG9wLW9wYWNpdHk9Ii4xIi8+PC9saW5lYXJHcmFkaWVudD48Y2xpcFBhdGggaWQ9InIiPjxyZWN0IHdpZHRoPSI5MCIgaGVpZ2h0PSIyMCIgcng9IjMiIGZpbGw9IiNmZmYiLz48L2NsaXBQYXRoPjxnIGNsaXAtcGF0aD0idXJsKCNyKSI+PHJlY3Qgd2lkdGg9IjMzIiBoZWlnaHQ9IjIwIiBmaWxsPSIjNTU1Ii8+PHJlY3QgeD0iMzMiIHdpZHRoPSI1NyIgaGVpZ2h0PSIyMCIgZmlsbD0iI2UwNWQ0NCIvPjxyZWN0IHdpZHRoPSI5MCIgaGVpZ2h0PSIyMCIgZmlsbD0idXJsKCNzKSIvPjwvZz48ZyBmaWxsPSIjZmZmIiB0ZXh0LWFuY2hvcj0ibWlkZGxlIiBmb250LWZhbWlseT0iVmVyZGFuYSxHZW5ldmEsRGVqYVZ1IFNhbnMsc2Fucy1zZXJpZiIgdGV4dC1yZW5kZXJpbmc9Imdlb21ldHJpY1ByZWNpc2lvbiIgZm9udC1zaXplPSIxMTAiPjx0ZXh0IGFyaWEtaGlkZGVuPSJ0cnVlIiB4PSIxNzUiIHk9IjE1MCIgZmlsbD0iIzAxMDEwMSIgZmlsbC1vcGFjaXR5PSIuMyIgdHJhbnNmb3JtPSJzY2FsZSguMSkiIHRleHRMZW5ndGg9IjIzMCI+Y2hhdDwvdGV4dD48dGV4dCB4PSIxNzUiIHk9IjE0MCIgdHJhbnNmb3JtPSJzY2FsZSguMSkiIGZpbGw9IiNmZmYiIHRleHRMZW5ndGg9IjIzMCI+Y2hhdDwvdGV4dD48dGV4dCBhcmlhLWhpZGRlbj0idHJ1ZSIgeD0iNjA1IiB5PSIxNTAiIGZpbGw9IiMwMTAxMDEiIGZpbGwtb3BhY2l0eT0iLjMiIHRyYW5zZm9ybT0ic2NhbGUoLjEpIiB0ZXh0TGVuZ3RoPSI0NzAiPm9uIFNsYWNrPC90ZXh0Pjx0ZXh0IHg9IjYwNSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iNDcwIj5vbiBTbGFjazwvdGV4dD48L2c+PC9zdmc+)](https://slack.x64dbg.com) [![Gitter](data:image/;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI5MiIgaGVpZ2h0PSIyMCIgcm9sZT0iaW1nIiBhcmlhLWxhYmVsPSJjaGF0OiBvbiBHaXR0ZXIiPjx0aXRsZT5jaGF0OiBvbiBHaXR0ZXI8L3RpdGxlPjxsaW5lYXJHcmFkaWVudCBpZD0icyIgeDI9IjAiIHkyPSIxMDAlIj48c3RvcCBvZmZzZXQ9IjAiIHN0b3AtY29sb3I9IiNiYmIiIHN0b3Atb3BhY2l0eT0iLjEiLz48c3RvcCBvZmZzZXQ9IjEiIHN0b3Atb3BhY2l0eT0iLjEiLz48L2xpbmVhckdyYWRpZW50PjxjbGlwUGF0aCBpZD0iciI+PHJlY3Qgd2lkdGg9IjkyIiBoZWlnaHQ9IjIwIiByeD0iMyIgZmlsbD0iI2ZmZiIvPjwvY2xpcFBhdGg+PGcgY2xpcC1wYXRoPSJ1cmwoI3IpIj48cmVjdCB3aWR0aD0iMzMiIGhlaWdodD0iMjAiIGZpbGw9IiM1NTUiLz48cmVjdCB4PSIzMyIgd2lkdGg9IjU5IiBoZWlnaHQ9IjIwIiBmaWxsPSJsaWdodHNlYWdyZWVuIi8+PHJlY3Qgd2lkdGg9IjkyIiBoZWlnaHQ9IjIwIiBmaWxsPSJ1cmwoI3MpIi8+PC9nPjxnIGZpbGw9IiNmZmYiIHRleHQtYW5jaG9yPSJtaWRkbGUiIGZvbnQtZmFtaWx5PSJWZXJkYW5hLEdlbmV2YSxEZWphVnUgU2FucyxzYW5zLXNlcmlmIiB0ZXh0LXJlbmRlcmluZz0iZ2VvbWV0cmljUHJlY2lzaW9uIiBmb250LXNpemU9IjExMCI+PHRleHQgYXJpYS1oaWRkZW49InRydWUiIHg9IjE3NSIgeT0iMTUwIiBmaWxsPSIjMDEwMTAxIiBmaWxsLW9wYWNpdHk9Ii4zIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgdGV4dExlbmd0aD0iMjMwIj5jaGF0PC90ZXh0Pjx0ZXh0IHg9IjE3NSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iMjMwIj5jaGF0PC90ZXh0Pjx0ZXh0IGFyaWEtaGlkZGVuPSJ0cnVlIiB4PSI2MTUiIHk9IjE1MCIgZmlsbD0iIzAxMDEwMSIgZmlsbC1vcGFjaXR5PSIuMyIgdHJhbnNmb3JtPSJzY2FsZSguMSkiIHRleHRMZW5ndGg9IjQ5MCI+b24gR2l0dGVyPC90ZXh0Pjx0ZXh0IHg9IjYxNSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iNDkwIj5vbiBHaXR0ZXI8L3RleHQ+PC9nPjwvc3ZnPg==)](https://gitter.im/x64dbg/x64dbg) [![Matrix](data:image/;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI5NCIgaGVpZ2h0PSIyMCIgcm9sZT0iaW1nIiBhcmlhLWxhYmVsPSJjaGF0OiBvbiBNYXRyaXgiPjx0aXRsZT5jaGF0OiBvbiBNYXRyaXg8L3RpdGxlPjxsaW5lYXJHcmFkaWVudCBpZD0icyIgeDI9IjAiIHkyPSIxMDAlIj48c3RvcCBvZmZzZXQ9IjAiIHN0b3AtY29sb3I9IiNiYmIiIHN0b3Atb3BhY2l0eT0iLjEiLz48c3RvcCBvZmZzZXQ9IjEiIHN0b3Atb3BhY2l0eT0iLjEiLz48L2xpbmVhckdyYWRpZW50PjxjbGlwUGF0aCBpZD0iciI+PHJlY3Qgd2lkdGg9Ijk0IiBoZWlnaHQ9IjIwIiByeD0iMyIgZmlsbD0iI2ZmZiIvPjwvY2xpcFBhdGg+PGcgY2xpcC1wYXRoPSJ1cmwoI3IpIj48cmVjdCB3aWR0aD0iMzMiIGhlaWdodD0iMjAiIGZpbGw9IiM1NTUiLz48cmVjdCB4PSIzMyIgd2lkdGg9IjYxIiBoZWlnaHQ9IjIwIiBmaWxsPSIjYTRhNjFkIi8+PHJlY3Qgd2lkdGg9Ijk0IiBoZWlnaHQ9IjIwIiBmaWxsPSJ1cmwoI3MpIi8+PC9nPjxnIGZpbGw9IiNmZmYiIHRleHQtYW5jaG9yPSJtaWRkbGUiIGZvbnQtZmFtaWx5PSJWZXJkYW5hLEdlbmV2YSxEZWphVnUgU2FucyxzYW5zLXNlcmlmIiB0ZXh0LXJlbmRlcmluZz0iZ2VvbWV0cmljUHJlY2lzaW9uIiBmb250LXNpemU9IjExMCI+PHRleHQgYXJpYS1oaWRkZW49InRydWUiIHg9IjE3NSIgeT0iMTUwIiBmaWxsPSIjMDEwMTAxIiBmaWxsLW9wYWNpdHk9Ii4zIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgdGV4dExlbmd0aD0iMjMwIj5jaGF0PC90ZXh0Pjx0ZXh0IHg9IjE3NSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iMjMwIj5jaGF0PC90ZXh0Pjx0ZXh0IGFyaWEtaGlkZGVuPSJ0cnVlIiB4PSI2MjUiIHk9IjE1MCIgZmlsbD0iIzAxMDEwMSIgZmlsbC1vcGFjaXR5PSIuMyIgdHJhbnNmb3JtPSJzY2FsZSguMSkiIHRleHRMZW5ndGg9IjUxMCI+b24gTWF0cml4PC90ZXh0Pjx0ZXh0IHg9IjYyNSIgeT0iMTQwIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgZmlsbD0iI2ZmZiIgdGV4dExlbmd0aD0iNTEwIj5vbiBNYXRyaXg8L3RleHQ+PC9nPjwvc3ZnPg==)](https://riot.im/app/#/room/#x64dbg:matrix.org) [![IRC](data:image/;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSI4MCIgaGVpZ2h0PSIyMCIgcm9sZT0iaW1nIiBhcmlhLWxhYmVsPSJjaGF0OiBvbiBJUkMiPjx0aXRsZT5jaGF0OiBvbiBJUkM8L3RpdGxlPjxsaW5lYXJHcmFkaWVudCBpZD0icyIgeDI9IjAiIHkyPSIxMDAlIj48c3RvcCBvZmZzZXQ9IjAiIHN0b3AtY29sb3I9IiNiYmIiIHN0b3Atb3BhY2l0eT0iLjEiLz48c3RvcCBvZmZzZXQ9IjEiIHN0b3Atb3BhY2l0eT0iLjEiLz48L2xpbmVhckdyYWRpZW50PjxjbGlwUGF0aCBpZD0iciI+PHJlY3Qgd2lkdGg9IjgwIiBoZWlnaHQ9IjIwIiByeD0iMyIgZmlsbD0iI2ZmZiIvPjwvY2xpcFBhdGg+PGcgY2xpcC1wYXRoPSJ1cmwoI3IpIj48cmVjdCB3aWR0aD0iMzMiIGhlaWdodD0iMjAiIGZpbGw9IiM1NTUiLz48cmVjdCB4PSIzMyIgd2lkdGg9IjQ3IiBoZWlnaHQ9IjIwIiBmaWxsPSJwdXJwbGUiLz48cmVjdCB3aWR0aD0iODAiIGhlaWdodD0iMjAiIGZpbGw9InVybCgjcykiLz48L2c+PGcgZmlsbD0iI2ZmZiIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZm9udC1mYW1pbHk9IlZlcmRhbmEsR2VuZXZhLERlamFWdSBTYW5zLHNhbnMtc2VyaWYiIHRleHQtcmVuZGVyaW5nPSJnZW9tZXRyaWNQcmVjaXNpb24iIGZvbnQtc2l6ZT0iMTEwIj48dGV4dCBhcmlhLWhpZGRlbj0idHJ1ZSIgeD0iMTc1IiB5PSIxNTAiIGZpbGw9IiMwMTAxMDEiIGZpbGwtb3BhY2l0eT0iLjMiIHRyYW5zZm9ybT0ic2NhbGUoLjEpIiB0ZXh0TGVuZ3RoPSIyMzAiPmNoYXQ8L3RleHQ+PHRleHQgeD0iMTc1IiB5PSIxNDAiIHRyYW5zZm9ybT0ic2NhbGUoLjEpIiBmaWxsPSIjZmZmIiB0ZXh0TGVuZ3RoPSIyMzAiPmNoYXQ8L3RleHQ+PHRleHQgYXJpYS1oaWRkZW49InRydWUiIHg9IjU1NSIgeT0iMTUwIiBmaWxsPSIjMDEwMTAxIiBmaWxsLW9wYWNpdHk9Ii4zIiB0cmFuc2Zvcm09InNjYWxlKC4xKSIgdGV4dExlbmd0aD0iMzcwIj5vbiBJUkM8L3RleHQ+PHRleHQgeD0iNTU1IiB5PSIxNDAiIHRyYW5zZm9ybT0ic2NhbGUoLjEpIiBmaWxsPSIjZmZmIiB0ZXh0TGVuZ3RoPSIzNzAiPm9uIElSQzwvdGV4dD48L2c+PC9zdmc+)](https://web.libera.chat/#x64dbg)

## ❤️ Sponsors
This project is partially made possible by the generous support of sponsors. We would like to give a special shout out to the following sponsors who donated at the highest tier since the last release:
- 🥇 [**FLOSS/fund**](https://floss.fund) - Funding for Free and Open Source projects.
- [**SEKTOR7 Institute**](https://institute.sektor7.net/)
- [**Back Engineering**](https://back.engineering)
- [**aslrnk**](https://aslr.ac)
- [**RedOps**](https://redops.at/en)

Also many thanks to the other active sponsors: @adam-the, @verdeckt, @emesare, @daaximus, @stevemk14ebr, @as0ni, @sedrakpc, @Dan0xE, @Invoke-RE, @leandrofroes, @shu-tom, @buzzer-re, @expend20, @crudd, @clayne, @fr0zenbag, @merces, @dzzie

**If you find x64dbg valuable in your work, please consider [becoming a sponsor](https://github.com/sponsors/mrexodia?metadata_source=releases). Your support directly funds development and helps us continue to build the future of debugging.** For companies we also offer custom services depending on the level of sponsorship, please [reach out](mailto:contact@x64dbg.com) for more details!

## 📜 Changelog
- Fix translation errors
- Split ModuleStdTable into header and sources to fix Q_OBJECT error
- Build with GitHub Actions by @mrexodia in #3538
- Ignore clangd cache
- Fix buffer overflow by @torusrxxx in #3541
- Add dump support for half float by @torusrxxx in #3544
- Add support for AVX512 by @torusrxxx in #3536
- Fix compilation on MinGW (clang) by @mrexodia in #3539
- Fix wrong icon for ActionTreatSelectionAsXXX by @torusrxxx in #3546
- Always update the Example text's background color by @G3ph4z in #3537
- Show launcher on first launch after installing by @whindsaks in #3521
- New x96dbg configuration dialog by @G3ph4z in #3532
- Make a rebuild copy the dependencies again
- Add headless mode POC #3516
- Fix compilation with clang-cl
- Add some more state for the headless mode
- Automatically download submodules when necessary
- Updated Type Database Handling by @snowsnowsnows in #3552
- Fix structure packing in TitanEngine.h #3563
- Update badges
- Improve plugin loading error messages
- Fix errors with simple header parsing
- Improved error reporting in DisplayType command
- Allow configuring default max pointer depth in DisplayType
- Small UX improvements in struct widget
- Move "Display type" to CommonActions and add to registers/stack
- Fix missing stdint types when building with vs2022 v120_xp toolset by @WindowsAPI in #3593
- Fix error handling in ParseTypes
- Added Copy Menu for Call Conv Arguments by @snowsnowsnows in #3595
- Draw jump lines under register label; Focus on window in handles view by @torusrxxx in #3585
- Copy graph image by @torusrxxx in #3578
- Add IRC link to README
- Added Type Reload Functionality in Struct View by @snowsnowsnows in #3597
- Update AStyleHelper to only format changed files
- Mark padding as bitfield
- Fix a bug with GotoDialog::setInitialExpression not working twice in a row
- Performance improvements for StructWidget
- Remove Visual Studio projects
- Static MSVC runtime for launcher and loaddll
- Prefix commands with $ to format the command itself
- Add a few string expression functions
- Update breakpoint dialog tooltip match reality
- Add test for #3599
- Add back missing /DELAYLOAD options #3412
- Add selectionChanged signal to SearchListView
- Implement DisplayTypeDialog for better UX
- Change StructWidget to inherit from TypeWidget
- Make type visiting expansion depth configurable
- Add GuiTypeListUpdated for better UX
- Hide __anonymous types
- Improve pretty printing in NodeVisitor
- Print string values of character arrays
- Get rid of Qt5Network dependency
- Match snapshot format to previous releases
- Update dependencies
- Upload snapshot ZIP files as artifacts directly
- Fix: Trim trailing whitespace in script labels #3589 by @oxqnd in #3610
- Fix Handles->Message Breakpoint: wnd proc or handle is unavailable (wine) by @eldarkg in #3607
- Fixed inverted logic by @OldGamesCracking in #3588


